<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Comment;

use Erlage\Photogram\System;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\Post\PostModelHelper;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\Post\PostCommentTable;
use Erlage\Photogram\Data\Dtos\User\UserMetaPushSettingsDTO;
use Erlage\Photogram\Data\Models\Notification\NotificationEnum;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentModel;
use Erlage\Photogram\Data\Tables\Notification\NotificationTable;
use Erlage\Photogram\Data\Models\Notification\NotificationFinder;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentBuilder;
use Erlage\Photogram\Data\Models\Notification\NotificationBuilder;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowFinder;
use Erlage\Photogram\Data\Models\Hashtag\Follow\HashtagFollowBuilder;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentModelHelper;
use Erlage\Photogram\Data\Dtos\Notification\NotificationLinkedContentDTO;

final class PostCommentActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | add comment
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | flags
            |--------------------------------------------------------------------------
            */

            $flagUpdateCache = false;

            $flagTweakUserFeed = false;

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postIdFromReq = self::$request -> findKey(
                PostCommentTable::PARENT_POST_ID,
                RequestTable::PAYLOAD,
                PostCommentTable::TABLE_NAME
            );

            $postCommentDisplayTextFromReq = self::$request -> findKey(
                PostCommentTable::DISPLAY_TEXT,
                RequestTable::PAYLOAD,
                PostCommentTable::TABLE_NAME
            );

            // optional
            $postCommentIdFromReq = self::$request -> findKey(
                PostCommentTable::REPLY_TO_POST_COMMENT_ID,
                RequestTable::PAYLOAD,
                PostCommentTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postIdFromReq);

            $replyToAPostComment = self::isAvailable($postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure post models exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure post comment model exists(if replying to a comment)
            |--------------------------------------------------------------------------
            */

            if ($replyToAPostComment)
            {
                // allow only one level comments to match with insta style xD
                // thou we can change this here easily by removing this block of code

                $commentIdToCheck = $postCommentIdFromReq;
                $maxWalkLength = 2;

                while (true)
                {
                    $commentIdToCheck = isset($targetPostCommentModel) ? $targetPostCommentModel -> getReplyToPostcommentId() : $postCommentIdFromReq;

                    $targetPostCommentModel = PostCommentModel::findFromId_throwException($commentIdToCheck);

                    // walk to level above
                    $commentIdToCheck = $targetPostCommentModel -> getReplyToPostCommentId();

                    if (0 == $commentIdToCheck)
                    {
                        // top level reached
                        break;
                    }

                    // it's not required but client can forge a requst to exhaust the system
                    // so we prevent that from happening here
                    if ($maxWalkLength-- < 0)
                    {
                        throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                    }
                }

                // make sure we're on the same page
                if ($targetPostModel -> getId() != $targetPostCommentModel -> getParentPostId())
                {
                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }
            }

            /*
            |--------------------------------------------------------------------------
            | ensure post owner exists
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks against post owner
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | build post comment
            |--------------------------------------------------------------------------
            */

            $postCommentModelBuilder = (new PostCommentBuilder())
                -> setParentPostId($targetPostModel -> getId())
                -> setOwnerUserId(self::$authedUserModel -> getId())
                -> setDisplayText($postCommentDisplayTextFromReq);

            /*
            |--------------------------------------------------------------------------
            | if replying to a post comment, add post comment id
            |--------------------------------------------------------------------------
            */

            if ($replyToAPostComment)
            {
                $postCommentModelBuilder -> setReplyToPostcommentId($targetPostCommentModel -> getId());
            }

            $postCommentModel = $postCommentModelBuilder -> dispense();

            /*
            |--------------------------------------------------------------------------
            | do post like
            |--------------------------------------------------------------------------
            */

            $postCommentModel -> save();

            /*
            |--------------------------------------------------------------------------
            |  if everything is alright, add data to reponse
            |--------------------------------------------------------------------------
            */

            self::addToResponse(PostCommentTable::getTableName(), $postCommentModel -> getDataMap());

            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());

            $flagUpdateCache = true;

            /*
            |--------------------------------------------------------------------------
            |  process flags
            |--------------------------------------------------------------------------
            */

            if ($flagUpdateCache)
            {
                /*
                |--------------------------------------------------------------------------
                | dispatch notifications
                |--------------------------------------------------------------------------
                */

                $notifyPostOwner = true;
                $notifyCommentOwner = $replyToAPostComment;

                /*
                |--------------------------------------------------------------------------
                | make sure we're not sending notifications to ourselves
                |--------------------------------------------------------------------------
                */

                if ($postOwnerUserModel -> getId() == self::$authedUserModel -> getId())
                {
                    $notifyPostOwner = false;
                }

                if ($replyToAPostComment && $targetPostCommentModel -> getOwnerUserId() == self::$authedUserModel -> getId())
                {
                    $notifyCommentOwner = false;
                }

                /*
                |--------------------------------------------------------------------------
                | check if post owner has notifications enabled
                |--------------------------------------------------------------------------
                */

                if ($notifyPostOwner)
                {
                    $metaPushSettings = $postOwnerUserModel -> getMetaPushSettings();

                    switch ($metaPushSettings -> getComments())
                    {
                        case UserMetaPushSettingsDTO::VALUE_OFF:
                            $notifyPostOwner = false;

                        break;

                        case UserMetaPushSettingsDTO::VALUE_FROM_PEOPLE_I_FOLLOW:
                            $notifyPostOwner = CommonQueries::isFollowing(
                                $postOwnerUserModel -> getId(),
                                self::$authedUserModel -> getId()
                            );

                        break;
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | check if comment owner has notifications enabled
                |--------------------------------------------------------------------------
                */

                if ($notifyCommentOwner)
                {
                    $targetPostCommentOwnerModel = UserModel::findFromId_noException($targetPostCommentModel -> getOwnerUserId());

                    if ($postOwnerUserModel -> isNotModel())
                    {
                        $notifyCommentOwner = false;
                    }
                    else
                    {
                        $metaPushSettings = $targetPostCommentOwnerModel -> getMetaPushSettings();

                        $notifyCommentOwner = UserMetaPushSettingsDTO::VALUE_ON == $metaPushSettings -> getCommentLikesAndPins();
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | notify post owner
                |--------------------------------------------------------------------------
                */

                if ($notifyPostOwner)
                {
                    /*
                    |--------------------------------------------------------------------------
                    | first try to find if someone already commented on the same post
                    |--------------------------------------------------------------------------
                    */

                    $finder = (new NotificationFinder())
                        -> setToUserId($postOwnerUserModel -> getId())
                        -> setTargetContentId($targetPostModel -> getId())
                        -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT)
                        -> setStampRegistrationDate(System::isoDate())
                        -> find();

                    if ($finder -> isFound())
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | notification exists, simply add user to existing notifcation
                        |--------------------------------------------------------------------------
                        */

                        $notifcationModel = $finder -> popModelFromResults();

                        $notificationContent = $notifcationModel -> getLinkedContent();

                        if ( ! $notificationContent -> getUserIds() -> containsValue(self::$authedUserModel -> getId()))
                        {
                            $notificationContent -> linkUserId(self::$authedUserModel -> getId());

                            $notifcationModel -> update(
                                array(
                                    NotificationTable::LINKED_CONTENT => $notificationContent,
                                    NotificationTable::META_IS_READ   => NotificationEnum::META_IS_READ_NO,
                                )
                            );

                            $notifcationModel -> save();
                        }
                    }
                    else
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | notification doesnt exists, create and dispatch a new notification
                        |--------------------------------------------------------------------------
                        */

                        $notificationContent = new NotificationLinkedContentDTO();

                        $notificationContent -> linkUserId(self::$authedUserModel -> getId());
                        $notificationContent -> linkPostId($targetPostModel -> getId());

                        $notificationModel = (new NotificationBuilder())
                            -> setToUserId($postOwnerUserModel -> getId())
                            -> setTargetContentId($targetPostModel -> getId())
                            -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT)
                            -> setLinkedContent($notificationContent)
                            -> dispense();

                        $notificationModel -> save();
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | notify comment owner
                |--------------------------------------------------------------------------
                */

                if ($notifyCommentOwner)
                {
                    /*
                    |--------------------------------------------------------------------------
                    | first try to find if someone already replied on same comment
                    |--------------------------------------------------------------------------
                    */

                    $finder = (new NotificationFinder())
                        -> setToUserId($targetPostCommentOwnerModel -> getId())
                        -> setTargetContentId($targetPostCommentModel -> getId())
                        -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT_REPLY)
                        -> setStampRegistrationDate(System::isoDate())
                        -> find();

                    if ($finder -> isFound())
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | notification exists, simply add user to existing notifcation
                        |--------------------------------------------------------------------------
                        */

                        $notifcationModel = $finder -> popModelFromResults();

                        $notificationContent = $notifcationModel -> getLinkedContent();

                        if ( ! $notificationContent -> getUserIds() -> containsValue(self::$authedUserModel -> getId()))
                        {
                            $notificationContent -> linkUserId(self::$authedUserModel -> getId());

                            $notifcationModel -> update(
                                array(
                                    NotificationTable::LINKED_CONTENT => $notificationContent,
                                    NotificationTable::META_IS_READ   => NotificationEnum::META_IS_READ_NO,
                                )
                            );

                            $notifcationModel -> save();
                        }
                    }
                    else
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | notification doesnt exists, create and dispatch a new notification
                        |--------------------------------------------------------------------------
                        */

                        $notificationContent = new NotificationLinkedContentDTO();

                        $notificationContent -> linkUserId(self::$authedUserModel -> getId());
                        $notificationContent -> linkPostId($targetPostModel -> getId());

                        $notificationModel = (new NotificationBuilder())
                            -> setToUserId($targetPostCommentOwnerModel -> getId())
                            -> setTargetContentId($targetPostCommentModel -> getId())
                            -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT_REPLY)
                            -> setLinkedContent($notificationContent)
                            -> dispense();

                        $notificationModel -> save();
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | notify users who are tagged in the post which just got liked
                |--------------------------------------------------------------------------
                */

                if (true)
                {
                    $targetPostDisplayContent = $targetPostModel -> getDisplayContent();

                    $targetPostDisplayContent -> iteratorPrepare();

                    /*
                    |--------------------------------------------------------------------------
                    | for all images in post content
                    |--------------------------------------------------------------------------
                    */

                    while ($targetPostDisplayContent -> iteratorForward())
                    {
                        $displayContentItemDTO = $targetPostDisplayContent -> iteratorCurrentDisplayContentItem();

                        /*
                        |--------------------------------------------------------------------------
                        | get tagged users DTO
                        |--------------------------------------------------------------------------
                        */

                        $taggedUsersDTO = $displayContentItemDTO -> getDisplayUserTagsOnItemDTO();

                        /*
                        |--------------------------------------------------------------------------
                        | for all tagged users
                        |--------------------------------------------------------------------------
                        */

                        $taggedUsersDTO -> iteratorPrepare();

                        while ($taggedUsersDTO -> iteratorForward())
                        {
                            $taggedUserDTO = $taggedUsersDTO -> iteratorCurrentDisplayUserTag();

                            /*
                            |--------------------------------------------------------------------------
                            | if tagged user is post owner then don't send them a notification
                            |--------------------------------------------------------------------------
                            */

                            if ($postOwnerUserModel -> getId() == $taggedUserDTO -> getUserId())
                            {
                                continue;
                            }

                            /*
                            |--------------------------------------------------------------------------
                            | if tagged user is current, don't send either
                            |--------------------------------------------------------------------------
                            */

                            if (self::$authedUserModel -> getId() == $taggedUserDTO -> getUserId())
                            {
                                continue;
                            }

                            /*
                            |--------------------------------------------------------------------------
                            | get tagged user model
                            |--------------------------------------------------------------------------
                            */

                            $taggedUserModel = UserModel::findFromId_noException($taggedUserDTO -> getUserId());

                            if ($taggedUserModel -> isModel())
                            {
                                $pushSettings = $taggedUserModel -> getMetaPushSettings();

                                /*
                                |--------------------------------------------------------------------------
                                | if tagged user has opted in for notifications
                                |--------------------------------------------------------------------------
                                */

                                if (UserMetaPushSettingsDTO::VALUE_ON == $pushSettings -> getCommentsOnPhotosOfYou())
                                {
                                    /*
                                    |--------------------------------------------------------------------------
                                    | first try to find if notification already exists
                                    |--------------------------------------------------------------------------
                                    */

                                    $finder = (new NotificationFinder())
                                        -> setToUserId($taggedUserModel -> getId())
                                        -> setTargetContentId($targetPostModel -> getId())
                                        -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT_ON_PHOTO_OF_YOU)
                                        -> setStampRegistrationDate(System::isoDate())
                                        -> find();

                                    if ($finder -> isFound())
                                    {
                                        /*
                                        |--------------------------------------------------------------------------
                                        | notification exists, simply add user to existing notifcation
                                        |--------------------------------------------------------------------------
                                        */

                                        $notifcationModel = $finder -> popModelFromResults();

                                        $notificationContent = $notifcationModel -> getLinkedContent();

                                        if ( ! $notificationContent -> getUserIds() -> containsValue(self::$authedUserModel -> getId()))
                                        {
                                            $notificationContent -> linkUserId(self::$authedUserModel -> getId());

                                            $notifcationModel -> update(
                                                array(
                                                    NotificationTable::LINKED_CONTENT => $notificationContent,
                                                    NotificationTable::META_IS_READ   => NotificationEnum::META_IS_READ_NO,
                                                )
                                            );

                                            $notifcationModel -> save();
                                        }
                                    }
                                    else
                                    {
                                        /*
                                        |--------------------------------------------------------------------------
                                        | notification doesnt exists, create and dispatch a new notification
                                        |--------------------------------------------------------------------------
                                        */

                                        $notificationContent = new NotificationLinkedContentDTO();

                                        $notificationContent -> linkUserId(self::$authedUserModel -> getId());
                                        $notificationContent -> linkPostId($targetPostModel -> getId());

                                        $notificationModel = (new NotificationBuilder())
                                            -> setToUserId($taggedUserModel -> getId())
                                            -> setTargetContentId($targetPostModel -> getId())
                                            -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT_ON_PHOTO_OF_YOU)
                                            -> setLinkedContent($notificationContent)
                                            -> dispense();

                                        $notificationModel -> save();
                                    }
                                }
                            }
                        }
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | user tweak realted flag setting
                |--------------------------------------------------------------------------
                */

                PostModelHelper::updateCacheCommentsCount($targetPostModel);

                if ($replyToAPostComment)
                {
                    PostCommentModelHelper::updateCacheCommentsCount($targetPostCommentModel);

                    $flagTweakUserFeed = Settings::getBool(ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_COMMENT);
                }
                else
                {
                    $flagTweakUserFeed = Settings::getBool(ServerConstants::SS_BOOL_IMPLICIT_HASHTAG_FOLLOW_ON_POST_COMMENT);
                }
            }

            /*
            |--------------------------------------------------------------------------
            | if a post comment and settings are enabled then tweak user feed
            |--------------------------------------------------------------------------
            */
            if ($flagTweakUserFeed)
            {
                $metaHashtags = $targetPostModel -> getMetaHashtags();

                $metaHashtags -> iteratorPrepare();

                while ($metaHashtags -> iteratorForward())
                {
                    $hashtagId = $metaHashtags -> iteratorCurrentHashtagId();

                    $finder = (new HashtagFollowFinder())
                        -> setFollowedHashtagId($hashtagId)
                        -> setFollowedByUserId(self::$authedUserModel -> getId())
                        -> find();

                    if ($finder -> notFound())
                    {
                        $hashtagFollowModel = (new HashtagFollowBuilder())
                            -> setFollowedHashtagId($hashtagId)
                            -> setFollowedByUserId(self::$authedUserModel -> getId())
                            -> dispense();

                        $hashtagFollowModel -> save();
                    }
                }
            }
        });
    }

    /*
    |--------------------------------------------------------------------------
    | remove comment
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | flags
            |--------------------------------------------------------------------------
            */

            $flagUpdateCache = false;

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postCommentIdFromReq = self::$request -> findKey(
                PostCommentTable::ID,
                RequestTable::PAYLOAD,
                PostCommentTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure post comment models exists
            |--------------------------------------------------------------------------
            */

            $targetPostcommentModel = PostCommentModel::findFromId_throwException($postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure post models exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($targetPostcommentModel -> getParentPostId());

            /*
            |--------------------------------------------------------------------------
            | ensure post owner exists
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks against post owner
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | delete
            |--------------------------------------------------------------------------
            */

            $targetPostcommentModel -> delete();

            $flagUpdateCache = true;

            /*
            |--------------------------------------------------------------------------
            |  process flags
            |--------------------------------------------------------------------------
            */

            if ($flagUpdateCache)
            {
                PostModelHelper::updateCacheCommentsCount($targetPostModel);

                // if it's a reply to a comment, update parent comment replies count
                if ('0' != $targetPostcommentModel -> getReplyToPostCommentId())
                {
                    $parentCommentModel = PostCommentModel::findFromId_noException($targetPostcommentModel -> getReplyToPostCommentId());

                    if ($parentCommentModel -> isModel())
                    {
                        PostCommentModelHelper::updateCacheCommentsCount($parentCommentModel);
                    }
                }
            }
        });
    }
}
